% This function takes in a set of frame numbers that have been
% hand-annotated by watching videos of pumping, that indicate when the
% contraction occurs.  It then plots the data.
%
% Note that if the worm moves too quickly certain frames of the video may
% not include the pharynx.  These frames are indicated in the 'missing'
% variable, and dealt with as if they were missing data.  This seems like
% the most reasonable way to deal with this data.
%
% Both pumpFrames and missingFrames are cell arrays, with each cell
% containing the data from 1 worm.
%
% start is the time to start analyzing, in seconds. 
% stop is the time to stop analyzing, in seconds.
% winWidth is the number of seconds over which to do the average.
% (DIFFERENT than other scripts, where it is in milliseconds)

function plotDataFromVideo(name, pumpFrames, missingFrames, name2, pumpFrames2, missingFrames2, fps, startInSec, stopInSec, winWidthInSec)

startDelay = 10;  % until stim on
stimLen = 10;
yMax = 7;
eventName = 'pumping rate (Hz)';
yBar = 6.5;

lightc = [98 0 255] ./ 255;  %VIOLET % [98 0 255] ./ 255; %BLUE          % Color of light bar
linecm = [0 0 0; 
          255 51 0;
          %0  0 255;
          ] ./ 255;  % Color of 2 mean lines
patchcm = [171 171 171; 
            255 214 126
           % 204 204 255;
            ] ./ 255;  % Color of 2 error regions around the mean

winWidthInFrames = floor(winWidthInSec * fps);
startFrame = floor(startInSec * fps) + 1;  % Add one so a start of 0 gets converted to frame #1
stopFrame = floor(stopInSec * fps - winWidthInFrames);
totalTime = stopInSec - startInSec;

% This is a cell vector, with each cell containing a vector where each
% value indicates the average pumping rate for that timepoint for one
% video.  This cell array will be analyze later to determine the mean and
% standard deviation to be plotted.
pumpRate = cell(1, stopFrame - startFrame + 1);

for fr = startFrame:stopFrame   % For each frame to analyze
    for vid = 1:length(pumpFrames)   % For each video to analyze
        % If the grinder is not missing from this frame in the video,, and
        % the whole window to be analyzed is clear of missing data
        if (isempty(find(missingFrames{vid} >= fr & missingFrames{vid} < fr+winWidthInFrames, 1)))  
            rate = length(find(pumpFrames{vid} >= fr & pumpFrames{vid} < fr+winWidthInFrames)) / winWidthInSec;
            pumpRate{fr-startFrame+1} = cat(1, pumpRate{fr-startFrame+1}, rate);
        end
    end
end

% Now that all the data is collated, calculate the mean pump rate over time
m = cellfun(@mean, pumpRate);
sd = cellfun(@std, pumpRate);
se = sd ./ sqrt(cellfun(@length, pumpRate));

if (~isempty(pumpFrames2))
    pumpRate2 = cell(1, stopFrame - startFrame + 1);
    for fr = startFrame:stopFrame   % For each frame to analyze
        for vid = 1:length(pumpFrames2)   % For each video to analyze
            % If the grinder is not missing from this frame in the video,, and
            % the whole window to be analyzed is clear of missing data
            if (isempty(find(missingFrames2{vid} >= fr & missingFrames2{vid} < fr+winWidthInFrames, 1)))  
                rate = length(find(pumpFrames2{vid} >= fr & pumpFrames2{vid} < fr+winWidthInFrames)) / winWidthInSec;
                pumpRate2{fr-startFrame+1} = cat(1, pumpRate2{fr-startFrame+1}, rate);
            end
        end
    end

    % Now that all the data is collated, calculate the mean pump rate over time
    m2 = cellfun(@mean, pumpRate2);
    sd2 = cellfun(@std, pumpRate2);
    se2 = sd ./ sqrt(cellfun(@length, pumpRate2));
end

figure;
hold on

x = 1:1/fps:totalTime;

patch([x x(end:-1:1)], [m+se m(end:-1:1)-se(end:-1:1)], patchcm(1,:),'LineStyle','none');
if (~isempty(pumpFrames2))
    patch([x x(end:-1:1)], [m2+se2 m2(end:-1:1)-se2(end:-1:1)], patchcm(2,:),'LineStyle','none');
end

h(1) = plot(x, m, 'LineWidth', 2, 'Color', linecm(1,:));
if (~isempty(pumpFrames2))
    h(2) = plot(x, m2, 'LineWidth', 2, 'Color', linecm(2,:));
end
h(3) = plot([startDelay startDelay+stimLen], [yBar yBar], 'LineWidth', 16, 'Color', lightc);

t = title(['VIDEO: ' eventName ' for ' name2 ' (mean \pm sem, n = ' ...
            num2str(length(pumpFrames)) ')'], 'FontWeight', 'bold', 'FontSize', 16);
if (~isempty(pumpFrames2))
    t = title(['VIDEO: ' eventName ' for ' name2 ' (mean \pm sem, n = ' ...
            num2str(length(pumpFrames2)) ')'], 'FontWeight', 'bold', 'FontSize', 16);
end
%set(t, 'interpreter', 'none');  % Turn off _ causing subscripts
xlabel('time (s)', 'FontWeight', 'Bold', 'FontSize', 12);
ylabel([eventName], 'FontWeight', 'Bold', 'FontSize', 12);
prettify();
axis([1 totalTime 0 yMax]);

yts = 0:1:yMax;
set(gca, 'YTick', yts); 

%legend(h, s1n, s2n, stimn);
text(15, yBar, 'light',...
     'FontWeight', 'bold', 'FontSize', 10, 'Color', [1 1 1], ...
     'HorizontalAlignment', 'center');
%text(12.5, yMax-.35, stimn, 'FontWeight', 'bold');

%{
if(s1m(end) > s2m(end))
    y1 = s1m(end) + 0.4;
    y2 = s2m(end) - 0.6;
else
    y1 = s1m(end) - 0.6;
    y2 = s2m(end) + 0.4;
end
text(x(end), y1, s1n, ...
     'FontWeight', 'bold', 'FontSize', 24, 'Color', linecm(1,:), ...
     'HorizontalAlignment', 'right', 'VerticalAlignment', 'bottom');
text(x(end), y2, s2n, ...
     'FontWeight', 'bold', 'FontSize', 24, 'Color', linecm(2,:), ...
     'HorizontalAlignment', 'right', 'VerticalAlignment', 'bottom');
%}
 
prettify();

%x = cell2mat(pumpRate);
%figure
%plot(x);
disp('Analysis complete.');

end